// background.js - Фоновый скрипт для управления плагином
class BackgroundManager {
    constructor() {
        this.init();
    }

    init() {
        console.log('A-Pass Key background script initialized');
        this.setupEventListeners();
    }

    setupEventListeners() {
        // Обработчик установки расширения
        chrome.runtime.onInstalled.addListener((details) => {
            this.handleInstallation(details);
        });

        // Обработчик сообщений от content scripts
        chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
            return this.handleMessage(message, sender, sendResponse);
        });
    }

    handleInstallation(details) {
        console.log('Extension installed:', details.reason);
        
        if (details.reason === 'install') {
            this.initializeStorage();
            this.showWelcomeNotification();
        }
    }

    initializeStorage() {
        chrome.storage.local.set({
            userStorage: {
                nodes: {},
            }
        });
    }

    showWelcomeNotification() {
        chrome.notifications.create('welcome', {
            type: 'basic',
            iconUrl: chrome.runtime.getURL('icons/icon128.png'),
            title: 'A-Key',
            message: 'Extension Ready',
            priority: 1
        });
    }

    handleMessage(message, sender, sendResponse) {
        switch (message.action) {
            case 'getUser':
                this.getUser().then(userId => {
                    sendResponse({
                        success: true,
                        userId: userId,
                    });
                });
                return true;

            case 'updateUser':
                // console.log(message);
                this.updateUser(
                    message.data.nodeId,
                    message.data.userId,
                    message.data.userSecret
                ).then(() => {
                    sendResponse({ success: true });
                });
                return true;
            
            case 'generateProof':
                // console.log(message);
                this.generateProof(message.data.verifyId, message.data.verifyHash, message.data.verifyCode)
                    .then(result => sendResponse({ success: true, data: result }))
                    .catch(error => sendResponse({ success: false, error: error.message }));
                return true; 
            
            default:
                sendResponse({ success: false, error: 'Unknown action' });
                return false;
        }
    }

    getUser(node = 0) {
        return new Promise((resolve) => {
            chrome.storage.local.get(['userStorage'], (result) => {
                // console.log(result);
                if (node in result.userStorage.nodes) {
                    resolve(result.userStorage.nodes[node].userId);
                } else {
                    resolve(undefined);
                }
            });
        });
    }

    getUserCredentials(node = 0) {
        return new Promise((resolve) => {
            chrome.storage.local.get(['userStorage'], (result) => {
                if (node in result.userStorage.nodes) {
                    resolve(result.userStorage.nodes[node]);
                } else {
                    resolve(undefined);
                }
            });
        });
    }

    updateUser(nodeId = 0, userId, userSecret) {
        return new Promise((resolve) => {
            chrome.storage.local.get(['userStorage'], (result) => {
                const storage = result.userStorage;
                if (!('nodes' in storage)) {
                    storage.nodes = {};
                };
                storage.nodes[nodeId] = {
                    userId: userId,
                    userSecret: userSecret,
                };
                chrome.storage.local.set({ userStorage: storage }, () => {
                    resolve();
                });
            });
        });
    }

    // Функция для генерации доказательства
    async generateProof(node = 0, verifyHash, verifyCode) {
        try {
            // console.log('Начинаем генерацию доказательства...');
            
            const userCredentials = await this.getUserCredentials();

            if (userCredentials) {
                const input = {
                    userId: userCredentials.userId,
                    verifierId: node,
                    verifierHash: verifyHash,
                    nonce: verifyCode,
                    password: userCredentials.userSecret,
                };

                const wasmFile = chrome.runtime.getURL("circuits/apass.wasm");
                const zkeyFile = chrome.runtime.getURL("circuits/apass_final.zkey");

                const { proof, publicSignals } = await snarkjs.groth16.fullProve(
                    input,
                    wasmFile,
                    zkeyFile
                );
                
                // console.log('Доказательство сгенерировано успешно!');
                // console.log('Proof:', proof);
                // console.log('Public Signals:', publicSignals);
                
                return {
                    proof: proof,
                    publicSignals: publicSignals
                };
            }
            
        } catch (error) {
            console.error('Ошибка при генерации доказательства:', error);
            throw error;
        }
    }

}

// Инициализация фонового скрипта
const backgroundManager = new BackgroundManager();